import gobject, gtk3, gio, glib, gtksource
from  exampleapp import ExampleApp

type
  ExampleAppWindow* = ptr ExampleAppWindowObj
  ExampleAppWindowObj* = object of gtk3.ApplicationWindowObj

  ExampleAppWindowClass = ptr ExampleAppWindowClassObj
  ExampleAppWindowClassObj = object of gtk3.ApplicationWindowClassObj

  ExampleAppWindowPrivate = ptr  ExampleAppWindowPrivateObj
  ExampleAppWindowPrivateObj = object
    settings: gio.GSettings
    stack: gtk3.Stack
    search: Widget
    searchbar: gtk3.SearchBar
    searchentry: Entry
    gears: MenuButton
    sidebar: Widget
    words: Container
    lines: Widget
    linesLabel: Widget

template exampleAppWindowType*(): expr = exampleAppWindowGetType()

template exampleAppWindow*(obj: expr): expr =
  gTypeCheckInstanceCast(obj, exampleAppWindowType, ExampleAppWindowObj)

gDefineTypeWithPrivate(ExampleAppWindow, applicationWindowGetType())

const private = exampleAppWindowGetInstancePrivate

proc searchTextChanged(entry: Entry) {.cdecl.} =
  var
    win: ExampleAppWindow
    text: cstring
    view: TextView
    buffer: TextBuffer
    start, matchStart, matchEnd, dumm: TextIterObj

  text = entry.text
  if text[0] == '\0': return
  win = exampleAppWindow(entry.toplevel)
  view = textView(child(bin(visibleChild(private(win).stack))))
  buffer = view.buffer
  buffer.getStartIter(start)
  if forwardSearch(start, text, TextSearchFlags.CASE_INSENSITIVE,
                          matchStart, matchEnd, nil) == GTRUE:
    buffer.selectRange(matchStart, matchEnd)
    discard view.scrollToIter(matchStart, 0, false, 0, 0)

proc findWord(button: Button; win: ExampleAppWindow) {.cdecl.} =
  setText(private(win).searchentry, button.label)

proc updateWords(win: ExampleAppWindow) {.cdecl.} =
  var
    priv: ExampleAppWindowPrivate
    strings: glib.GHashTable
    iter: glib.GHashTableIterObj
    tab: Widget
    view: TextView
    row: Widget
    buffer: TextBuffer
    start, `end`: TextIterObj
    children, l: glib.GList
    word, key: cstring

  priv = win.private
  tab = priv.stack.visibleChild
  if tab == nil: return
  view = textview(child(bin(tab)))
  buffer = view.buffer
  strings = newHashTable(strHash, strEqual, free, nil)
  buffer.getStartIter(start)
  block done:
    while not isEnd(start):
      while not startsWord(start):
        if not forwardChar(start): break done
      `end` = start
      if not forwardWordEnd(`end`): break done
      word = buffer.text(start, `end`, false)
      discard add(strings, utf8Strdown(word, -1))
      free(word)
      start = `end`
  children = priv.words.children
  l = children
  while l != nil:
    priv.words.remove(widget(l.data))
    l = l.next
  free(children)
  init(iter, strings)
  while iter.next(cast[ptr Gpointer](addr(key)), nil):
    row = newButton(key)
    discard gSignalConnect(row, "clicked", gCallback(findWord), win)
    show(row)
    add(priv.words, row)
  unref(strings)

proc updateLines(win: ExampleAppWindow) {.cdecl.} =
  var
    priv: ExampleAppWindowPrivate
    tab: Widget
    buffer: TextBuffer
    iter: TextIterObj
    count: cint
    lines: cstring

  priv = private(win)
  tab = priv.stack.visibleChild
  if tab == nil: return
  buffer = textview(child(bin(tab))).buffer
  #buffer = gtk3.bin(tab)
  count = 0
  buffer.getStartIter(iter)
  while not isEnd(iter):
    inc(count)
    if not forwardLine(iter): break
  lines = dupPrintf("%d", count)
  setText(label(priv.lines), lines)
  free(lines)

proc visibleChildChanged(stack: GObject; pspec: GParamSpec) {.cdecl.} =
  var
    win: ExampleAppWindow
    priv: ExampleAppWindowPrivate
  if inDestruction(widget(stack)): return
  win = exampleAppWindow(toplevel(widget(stack)))
  priv = private(win)
  priv.searchbar.searchMode = false
  updateWords(win)
  updateLines(win)

proc wordsChanged(sidebar: GObject; pspec: GParamSpec; win: ExampleAppWindow) {.cdecl.} =
  updateWords(win)

proc exampleAppWindowInit(self: ExampleAppWindow) =
  var
    priv: ExampleAppWindowPrivate
    builder: Builder
    menu: gio.GMenuModel
    action: gio.GAction
    
  priv = private(self)
  initTemplate(self)
  priv.settings = newSettings("org.gtk.exampleapp")
  `bind`(priv.settings, "transition", priv.stack, "transition-type",
                  gio.GSettingsBindFlags.DEFAULT)
  `bind`(priv.settings, "show-words", priv.sidebar, "reveal-child",
                  gio.GSettingsBindFlags.DEFAULT)
  discard objectBindProperty(priv.search, "active", priv.searchbar,
                         "search-mode-enabled", gobject.GBindingFlags.BIDIRECTIONAL)
  discard gSignalConnect(priv.sidebar, "notify::reveal-child",
                   gCallback(wordsChanged), self)
  builder = newBuilder(resourcePath = "/org/gtk/exampleapp/gears-menu.ui")
  #builder = gtk3.builderNewFromFile("gears-menu.ui")
  menu = gMenuModel(getObject(builder, "menu"))
  setMenuModel(priv.gears, menu)
  objectUnref(builder)
  action = createAction(priv.settings, "show-words")
  addAction(gActionMap(self), action)
  objectUnref(action)
  action = cast[GAction](newPropertyAction("show-lines", priv.lines, "visible"))
  addAction(gActionMap(self), action)
  objectUnref(action)
  discard objectBindProperty(priv.lines, "visible", priv.linesLabel, "visible",
                         gobject.GBindingFlags.DEFAULT)
  objectSet(settingsGetDefault(), "gtk-shell-shows-app-menu", false, nil)
  setShowMenubar(self, true)

proc exampleAppWindowDispose(obj: GObject) {.cdecl.} =
  var
    win: ExampleAppWindow
    priv: ExampleAppWindowPrivate
  win = exampleAppWindow(obj)
  priv = win.private
  var hhh = GObject(priv.settings)
  clearObject(hhh)
  priv.settings = nil # https://github.com/nim-lang/Nim/issues/3449
  gObjectClass(exampleAppWindowParentClass).dispose(obj)

proc exampleAppWindowClassInit(klass: ExampleAppWindowClass) =
  klass.dispose = exampleAppWindowDispose
  setTemplateFromResource(klass, "/org/gtk/exampleapp/window.ui")
  # 
  # we may replace function call above by this code to avoid usage of resource:
  ###  var
  ###    buffer: cstring
  ###    length: gsize
  ###    error: glib.GError = nil
  ###    gbytes: glib.GBytes = nil

  ###  if not gFileGetContents("window.ui", buffer, length, error):
  ###    gCritical("Unable to load window.ui \'%s\': %s",
  ###               gObjectClassName(klass), error.message)
  ###    free(error)
  ###    return
  ###  gbytes = gBytesNew(buffer, length)
  ###  setTemplate(klass, gbytes)
  ###  gFree(buffer)
  ### done with replace
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, stack)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, search)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, searchbar)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, searchentry)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, gears)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, words)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, sidebar)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, lines)
  widgetClassBindTemplateChildPrivate(klass, ExampleAppWindow, linesLabel)
  widgetClassBindTemplateCallback(klass, searchTextChanged)
  widgetClassBindTemplateCallback(klass, visibleChildChanged)

proc exampleAppWindowNew*(app: ExampleApp): ExampleAppWindow {.cdecl.} =
  result = cast[ExampleAppWindow](newObject(exampleAppWindowType, "application", app, nil))

proc exampleAppWindowOpen*(win: ExampleAppWindow; file: gio.GFile) {.cdecl.} =
  var
    priv: ExampleAppWindowPrivate
    basename, contents: cstring
    scrolled: ScrolledWindow
    view: TextView
    #buffer: TextBuffer
    buffer: gtksource.Buffer
    length: Gsize
    lm: LanguageManager
    language: Language
    tag: gtk3.TextTag
    dummy: GError = nil
    startIter, endIter: TextIterObj

  priv = private(win)
  basename = getBasename(file)
  scrolled = newScrolledWindow(nil, nil)
  show(scrolled)
  scrolled.hexpand = true
  scrolled.vexpand = true
  #view = textViewNew()
  lm = languageManagerGetDefault()
  language = lm.guessLanguage("test.nim", nil)
  #buffer = gtksource.bufferNew(nil)
  buffer = gtksource.newBuffer(language)
  view = gtksource.newView(buffer)
  view.editable = true
  view.cursorVisible = true
  show(view)
  scrolled.add(view)
  addTitled(priv.stack, scrolled, basename, basename)
  #buffer = view.buffer
  if loadContents(file, nil, contents, length, nil, dummy):
    buffer.setText(contents, (cint) length)
    free(contents)
  tag = buffer.createTag(nil, nil)
  `bind`(priv.settings, "font", tag, "font", gio.GSettingsBindFlags.DEFAULT)
  buffer.getStartIter(startIter)
  buffer.getEndIter(endIter)
  buffer.applyTag(tag, startIter, endIter)
  free(basename)
  priv.search.sensitive = true
  updateWords(win)
  updateLines(win)

